const prisma = require('../config/database');

// Item Management
const createItem = async (req, res) => {
  try {
    const { code, name, category, unitPrice, stock, reorderLevel, description } = req.body;

    // Create the inventory item
    const item = await prisma.inventoryItem.create({
      data: {
        code,
        name,
        category,
        unitPrice: parseFloat(unitPrice),
        reorderLevel: parseInt(reorderLevel || 10),
        description,
        // Create initial stock record for the default location
        stocks: {
          create: [
            {
              quantity: parseInt(stock || 0),
              lastStockUpdate: new Date(),
              location: {
                connectOrCreate: {
                  where: { locationName: 'Main Warehouse' },
                  create: {
                    locationName: 'Main Warehouse',
                    description: 'Default warehouse location',
                    isActive: true
                  }
                }
              }
            }
          ]
        }
      },
      include: {
        stocks: {
          include: {
            location: true
          }
        }
      }
    });

    // Format the response to match frontend expectations
    const formattedItem = {
      id: item.id,
      code: item.code,
      name: item.name,
      category: item.category,
      unitPrice: parseFloat(item.unitPrice),
      stock: item.stocks.reduce((total, stock) => total + stock.quantity, 0),
      reorderLevel: item.reorderLevel,
      description: item.description || ''
    };

    res.status(201).json(formattedItem);
  } catch (error) {
    console.error('Error creating inventory item:', error);
    if (error.code === 'P2002') {
      return res.status(400).json({ error: 'Item code already exists' });
    }
    res.status(500).json({ error: error.message });
  }
};

const updateItem = async (req, res) => {
  try {
    const { id } = req.params;
    const { code, name, category, unitPrice, stock, reorderLevel, description } = req.body;

    // Get current item with stocks
    const currentItem = await prisma.inventoryItem.findUnique({
      where: { id: parseInt(id) },
      include: { stocks: true }
    });

    if (!currentItem) {
      return res.status(404).json({ error: 'Item not found' });
    }

    // Update the item
    const updatedItem = await prisma.inventoryItem.update({
      where: { id: parseInt(id) },
      data: {
        code,
        name,
        category,
        unitPrice: parseFloat(unitPrice),
        reorderLevel: parseInt(reorderLevel || 10),
        description
      },
      include: {
        stocks: {
          include: {
            location: true
          }
        }
      }
    });

    // Update stock if provided and if there's a main warehouse stock record
    if (stock !== undefined) {
      const mainStock = updatedItem.stocks.find(s => 
        s.location.locationName === 'Main Warehouse'
      );

      if (mainStock) {
        await prisma.inventoryStock.update({
          where: { id: mainStock.id },
          data: {
            quantity: parseInt(stock),
            lastStockUpdate: new Date()
          }
        });
      } else {
        // Create stock record if it doesn't exist
        await prisma.inventoryStock.create({
          data: {
            itemId: updatedItem.id,
            quantity: parseInt(stock),
            lastStockUpdate: new Date(),
            location: {
              connectOrCreate: {
                where: { locationName: 'Main Warehouse' },
                create: {
                  locationName: 'Main Warehouse',
                  description: 'Default warehouse location',
                  isActive: true
                }
              }
            }
          }
        });
      }

      // Refresh the item data to include updated stock
      const refreshedItem = await prisma.inventoryItem.findUnique({
        where: { id: parseInt(id) },
        include: {
          stocks: {
            include: {
              location: true
            }
          }
        }
      });

      // Format the response
      const formattedItem = {
        id: refreshedItem.id,
        code: refreshedItem.code,
        name: refreshedItem.name,
        category: refreshedItem.category,
        unitPrice: parseFloat(refreshedItem.unitPrice),
        stock: refreshedItem.stocks.reduce((total, stock) => total + stock.quantity, 0),
        reorderLevel: refreshedItem.reorderLevel,
        description: refreshedItem.description || ''
      };

      return res.json(formattedItem);
    }

    // Format the response if stock wasn't updated
    const formattedItem = {
      id: updatedItem.id,
      code: updatedItem.code,
      name: updatedItem.name,
      category: updatedItem.category,
      unitPrice: parseFloat(updatedItem.unitPrice),
      stock: updatedItem.stocks.reduce((total, stock) => total + stock.quantity, 0),
      reorderLevel: updatedItem.reorderLevel,
      description: updatedItem.description || ''
    };

    res.json(formattedItem);
  } catch (error) {
    console.error('Error updating inventory item:', error);
    if (error.code === 'P2025') {
      return res.status(404).json({ error: 'Item not found' });
    }
    if (error.code === 'P2002') {
      return res.status(400).json({ error: 'Item code already exists' });
    }
    res.status(500).json({ error: error.message });
  }
};

const getItem = async (req, res) => {
  try {
    const { id } = req.params;

    const item = await prisma.inventoryItem.findUnique({
      where: { id: parseInt(id) },
      include: {
        stocks: {
          include: {
            location: true
          }
        }
      }
    });

    if (!item) {
      return res.status(404).json({ error: 'Item not found' });
    }

    // Format the response to match frontend expectations
    const formattedItem = {
      id: item.id,
      code: item.code,
      name: item.name,
      category: item.category,
      unitPrice: parseFloat(item.unitPrice),
      stock: item.stocks.reduce((total, stock) => total + stock.quantity, 0),
      reorderLevel: item.reorderLevel,
      description: item.description || ''
    };

    res.json(formattedItem);
  } catch (error) {
    console.error('Error fetching inventory item:', error);
    res.status(500).json({ error: error.message });
  }
};

const getAllItems = async (req, res) => {
  try {
    const items = await prisma.inventoryItem.findMany({
      include: {
        stocks: {
          include: {
            location: true
          }
        }
      },
      orderBy: {
        name: 'asc'
      }
    });

    // Format items to match frontend expectations
    const formattedItems = items.map(item => ({
      id: item.id,
      code: item.code,
      name: item.name,
      category: item.category,
      unitPrice: parseFloat(item.unitPrice),
      stock: item.stocks.reduce((total, stock) => total + stock.quantity, 0),
      reorderLevel: item.reorderLevel,
      description: item.description || ''
    }));

    res.json(formattedItems);
  } catch (error) {
    console.error('Error fetching inventory items:', error);
    res.status(500).json({ error: error.message });
  }
};

// Location Management
const createLocation = async (req, res) => {
  try {
    const { locationName, description } = req.body;

    const location = await prisma.inventoryLocation.create({
      data: {
        locationName,
        description
      }
    });

    res.status(201).json(location);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updateLocation = async (req, res) => {
  try {
    const { id } = req.params;
    const { locationName, description, isActive } = req.body;

    const location = await prisma.inventoryLocation.update({
      where: { id: parseInt(id) },
      data: {
        locationName,
        description,
        isActive
      }
    });

    res.json(location);
  } catch (error) {
    if (error.code === 'P2025') {
      return res.status(404).json({ error: 'Location not found' });
    }
    res.status(500).json({ error: error.message });
  }
};

const getLocation = async (req, res) => {
  try {
    const { id } = req.params;

    const location = await prisma.inventoryLocation.findUnique({
      where: { id: parseInt(id) },
      include: {
        stocks: {
          include: {
            item: true
          }
        }
      }
    });

    if (!location) {
      return res.status(404).json({ error: 'Location not found' });
    }

    res.json(location);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getAllLocations = async (req, res) => {
  try {
    const { isActive } = req.query;

    let whereClause = {};
    if (isActive !== undefined) {
      whereClause.isActive = isActive === 'true';
    }

    const locations = await prisma.inventoryLocation.findMany({
      where: whereClause,
      orderBy: {
        locationName: 'asc'
      }
    });

    res.json(locations);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

// Stock Management
const updateStock = async (req, res) => {
  try {
    const { itemId, locationId } = req.params;
    const { quantity, minimumQuantity, maximumQuantity } = req.body;

    const stock = await prisma.inventoryStock.upsert({
      where: {
        itemId_locationId: {
          itemId: parseInt(itemId),
          locationId: parseInt(locationId)
        }
      },
      update: {
        quantity,
        minimumQuantity,
        maximumQuantity,
        lastStockUpdate: new Date()
      },
      create: {
        itemId: parseInt(itemId),
        locationId: parseInt(locationId),
        quantity,
        minimumQuantity,
        maximumQuantity,
        lastStockUpdate: new Date()
      },
      include: {
        item: true,
        location: true
      }
    });

    res.json(stock);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getStock = async (req, res) => {
  try {
    const { itemId, locationId } = req.params;

    const stock = await prisma.inventoryStock.findUnique({
      where: {
        itemId_locationId: {
          itemId: parseInt(itemId),
          locationId: parseInt(locationId)
        }
      },
      include: {
        item: true,
        location: true
      }
    });

    if (!stock) {
      return res.status(404).json({ error: 'Stock not found' });
    }

    res.json(stock);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getLowStock = async (req, res) => {
  try {
    const lowStock = await prisma.inventoryStock.findMany({
      where: {
        quantity: {
          lte: prisma.raw('minimum_quantity')
        }
      },
      include: {
        item: true,
        location: true
      },
      orderBy: {
        quantity: 'asc'
      }
    });

    res.json(lowStock);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

module.exports = {
  createItem,
  updateItem,
  getItem,
  getAllItems,
  createLocation,
  updateLocation,
  getLocation,
  getAllLocations,
  updateStock,
  getStock,
  getLowStock
};
