const prisma = require('../config/database');

const createSupplier = async (req, res) => {
  try {
    const {
      name,
      supplierType,
      contactPerson,
      phone,
      email,
      address,
      taxId
    } = req.body;

    const supplier = await prisma.supplier.create({
      data: {
        name,
        supplierType,
        contactPerson,
        phone,
        email,
        address,
        taxId,
        account: {
          create: {}
        }
      },
      include: {
        account: true
      }
    });

    res.status(201).json(supplier);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updateSupplier = async (req, res) => {
  try {
    const { id } = req.params;
    const {
      name,
      supplierType,
      contactPerson,
      phone,
      email,
      address,
      taxId,
      isActive
    } = req.body;

    const supplier = await prisma.supplier.update({
      where: { id: parseInt(id) },
      data: {
        name,
        supplierType,
        contactPerson,
        phone,
        email,
        address,
        taxId,
        isActive
      },
      include: {
        account: true
      }
    });

    res.json(supplier);
  } catch (error) {
    if (error.code === 'P2025') {
      return res.status(404).json({ error: 'Supplier not found' });
    }
    res.status(500).json({ error: error.message });
  }
};

const getSupplier = async (req, res) => {
  try {
    const { id } = req.params;

    const supplier = await prisma.supplier.findUnique({
      where: { id: parseInt(id) },
      include: {
        account: true
      }
    });

    if (!supplier) {
      return res.status(404).json({ error: 'Supplier not found' });
    }

    res.json(supplier);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const getAllSuppliers = async (req, res) => {
  try {
    const { type, isActive, search } = req.query;

    let whereClause = {};

    if (type) {
      whereClause.supplierType = type;
    }

    if (isActive !== undefined) {
      whereClause.isActive = isActive === 'true';
    }

    if (search) {
      whereClause.OR = [
        { name: { contains: search, mode: 'insensitive' } },
        { contactPerson: { contains: search, mode: 'insensitive' } },
        { email: { contains: search, mode: 'insensitive' } }
      ];
    }

    const suppliers = await prisma.supplier.findMany({
      where: whereClause,
      include: {
        account: true
      },
      orderBy: {
        name: 'asc'
      }
    });

    res.json(suppliers);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

const updateSupplierAccount = async (req, res) => {
  try {
    const { supplierId } = req.params;
    const { totalPurchases, balanceDue } = req.body;

    const account = await prisma.supplierAccount.update({
      where: { supplierId: parseInt(supplierId) },
      data: {
        totalPurchases,
        balanceDue,
        lastTransactionDate: new Date()
      }
    });

    res.json(account);
  } catch (error) {
    if (error.code === 'P2025') {
      return res.status(404).json({ error: 'Supplier account not found' });
    }
    res.status(500).json({ error: error.message });
  }
};

const getSupplierAccount = async (req, res) => {
  try {
    const { supplierId } = req.params;

    const account = await prisma.supplierAccount.findUnique({
      where: { supplierId: parseInt(supplierId) },
      include: {
        supplier: true
      }
    });

    if (!account) {
      return res.status(404).json({ error: 'Supplier account not found' });
    }

    res.json(account);
  } catch (error) {
    res.status(500).json({ error: error.message });
  }
};

module.exports = {
  createSupplier,
  updateSupplier,
  getSupplier,
  getAllSuppliers,
  updateSupplierAccount,
  getSupplierAccount
};
