const { body } = require('express-validator');
const { validateRequest } = require('./validationMiddleware');
const prisma = require('../config/database');

const validateLogin = [
  body('username')
    .trim()
    .notEmpty()
    .withMessage('Username is required')
    .isLength({ min: 3, max: 50 })
    .withMessage('Username must be between 3 and 50 characters'),
  
  body('password')
    .trim()
    .notEmpty()
    .withMessage('Password is required')
    .isLength({ min: 6 })
    .withMessage('Password must be at least 6 characters'),
  
  validateRequest
];

const validateRegistration = [
  body('username')
    .trim()
    .notEmpty()
    .withMessage('Username is required')
    .isLength({ min: 3, max: 50 })
    .withMessage('Username must be between 3 and 50 characters')
    .custom(async (value) => {
      const existingUser = await prisma.user.findUnique({
        where: { username: value }
      });
      if (existingUser) {
        throw new Error('Username already exists');
      }
      return true;
    }),
  
  body('password')
    .trim()
    .notEmpty()
    .withMessage('Password is required')
    .isLength({ min: 6 })
    .withMessage('Password must be at least 6 characters'),
  
  body('fullName')
    .trim()
    .notEmpty()
    .withMessage('Full name is required')
    .isLength({ min: 2, max: 100 })
    .withMessage('Full name must be between 2 and 100 characters'),
  
  body('email')
    .trim()
    .optional({ nullable: true })
    .isEmail()
    .withMessage('Must be a valid email address')
    .custom(async (value) => {
      if (!value) return true;
      
      const existingUser = await prisma.user.findUnique({
        where: { email: value }
      });
      if (existingUser) {
        throw new Error('Email already in use');
      }
      return true;
    }),
  
  body('phone')
    .trim()
    .optional({ nullable: true })
    .isLength({ min: 10, max: 20 })
    .withMessage('Phone number must be between 10 and 20 characters'),
  
  validateRequest
];

const validateUserApproval = [
  body('userId')
    .isInt({ min: 1 })
    .withMessage('Valid user ID is required'),
  
  body('locationId')
    .isInt({ min: 1 })
    .withMessage('Valid location ID is required'),
  
  body('userRole')
    .isIn(['ADMIN', 'MANAGER', 'CASHIER', 'ACCOUNTANT'])
    .withMessage('Valid user role is required'),
  
  body('verificationStatus')
    .isIn(['APPROVED', 'REJECTED'])
    .withMessage('Verification status must be either APPROVED or REJECTED'),
  
  body('verificationNote')
    .optional({ nullable: true })
    .isString()
    .isLength({ max: 255 })
    .withMessage('Verification note must not exceed 255 characters'),
  
  validateRequest
];

module.exports = {
  validateLogin,
  validateRegistration,
  validateUserApproval
};
