const { body, param } = require('express-validator');
const { validateRequest } = require('./validationMiddleware');

const validateRegistration = [
  body('username')
    .trim()
    .notEmpty()
    .withMessage('Username is required')
    .isLength({ min: 3, max: 50 })
    .withMessage('Username must be between 3 and 50 characters')
    .matches(/^[a-zA-Z0-9_]+$/)
    .withMessage('Username can only contain letters, numbers, and underscores'),
  
  body('password')
    .trim()
    .notEmpty()
    .withMessage('Password is required')
    .isLength({ min: 6 })
    .withMessage('Password must be at least 6 characters')
    .matches(/^(?=.*[A-Za-z])(?=.*\d)[A-Za-z\d@$!%*#?&]+$/)
    .withMessage('Password must contain at least one letter and one number'),
  
  body('fullName')
    .trim()
    .notEmpty()
    .withMessage('Full name is required')
    .isLength({ max: 100 })
    .withMessage('Full name must be at most 100 characters'),
  
  body('email')
    .trim()
    .notEmpty()
    .withMessage('Email is required')
    .isEmail()
    .withMessage('Invalid email format')
    .isLength({ max: 100 })
    .withMessage('Email must be at most 100 characters'),
  
  body('phone')
    .optional()
    .trim()
    .isLength({ max: 20 })
    .withMessage('Phone must be at most 20 characters'),
  
  body('locationId')
    .notEmpty()
    .withMessage('Location is required')
    .isInt()
    .withMessage('Invalid location ID'),
  
  validateRequest
];

const validateVerification = [
  param('userId')
    .isInt()
    .withMessage('Invalid user ID'),
  
  body('status')
    .isIn(['APPROVED', 'REJECTED'])
    .withMessage('Status must be either APPROVED or REJECTED'),
  
  body('note')
    .optional()
    .trim()
    .isLength({ max: 255 })
    .withMessage('Note must be at most 255 characters'),
  
  validateRequest
];

module.exports = {
  validateRegistration,
  validateVerification
};
